#!/bin/bash

# Define variables
split_dir="./split_name_files/logo"          # Directory containing split files
file_pattern="dirs_part_*.txt"                # txt file contains the names of files remaminging to be proceessed
output_prefix="kmap_job"                     # Prefix for job names
log_dir="./job_logs"                         # Directory to store log files
err_dir="./job_errors"                       # Directory to store error files

# Resource configuration
mem_per_cpu="10G"                            # Memory per CPU
cpus_per_task="1"                            # Number of CPUs per task
time_limit="10:00:00"                        # Job time limit (format: HH:MM:SS)

# Create log and error directories if they don't exist
mkdir -p "$log_dir"
mkdir -p "$err_dir"

# Find all files matching the pattern and sort by name
split_files=($(ls "$split_dir"/$file_pattern 2>/dev/null | sort))

# Check if any split files were found
if [ ${#split_files[@]} -eq 0 ]; then
    echo "No matching split files found: $split_dir/$file_pattern"
    exit 1
fi

echo "Found ${#split_files[@]} split files."

# Loop over each split file and submit a job
index=1
for split_file in "${split_files[@]}"; do
    # Format the index as a three-digit number with leading zeros
    part_number=$(printf "%03d" "$index")
    
    # Define job name, log file, and error file
    job_name="${output_prefix}_${part_number}"
    log_file="${log_dir}/${job_name}.out"
    err_file="${err_dir}/${job_name}.err"
    
    # Submit the job
    sbatch --job-name="$job_name" \
           --output="$log_file" \
           --error="$err_file" \
           --ntasks=1 \
           --cpus-per-task="$cpus_per_task" \
           --mem-per-cpu="$mem_per_cpu" \
           --time="$time_limit" \
           ./run_kmap_pac_part.sh "$split_file"
    
    echo "Job submitted: $job_name, processing file: $split_file"
    
    # Increment index
    index=$((index + 1))
done

echo "All jobs have been submitted."
